import pandas as pd
import numpy as np
from sklearn.model_selection import StratifiedKFold
from sklearn.metrics import accuracy_score, precision_score, recall_score, f1_score

from gaussian import GNB
from kmedoid import k_medoids

df = pd.read_csv('./data_file/student_habits_performance.csv')
df['pass'] = (df['exam_score'] >= 60).astype(int)

features = [
    'study_hours_per_day',
    'social_media_hours',
    'netflix_hours',
    'attendance_percentage',
    'sleep_hours',
    'exercise_frequency',
    'mental_health_rating'
]
X = df[features].fillna(df[features].mean()).values
y = df['pass'].values

# Global 5-fold cross-validation
skf = StratifiedKFold(n_splits=5, shuffle=True, random_state=42)
global_metrics = {'Accuracy': [], 'Precision': [], 'Recall': [], 'F1': []}

for train_idx, test_idx in skf.split(X, y):
    X_train, X_test = X[train_idx], X[test_idx]
    y_train, y_test = y[train_idx], y[test_idx]

    clf = GNB()
    clf.fit(X_train, y_train)
    y_pred = clf.predict(X_test)

    global_metrics['Accuracy'].append(accuracy_score(y_test, y_pred))
    global_metrics['Precision'].append(precision_score(y_test, y_pred))
    global_metrics['Recall'].append(recall_score(y_test, y_pred))
    global_metrics['F1'].append(f1_score(y_test, y_pred))

global_summary = {metric: np.mean(scores) 
                  for metric, scores in global_metrics.items()}

# Cluster with k-medoids and cluster-wise 3-fold CV

# cluster_X = df[['netflix_hours']].fillna(df['netflix_hours'].mean()).values
labels, _ = k_medoids(X, k=5, random_state=42)
df['cluster'] = labels

cluster_summaries = {}
for c in sorted(np.unique(labels)):
    mask = labels == c
    Xc, yc = X[mask], y[mask]

    # Skip clusters with too few samples or only one class
    if len(yc) < 10 or len(np.unique(yc)) < 2:
        continue

    skf_c = StratifiedKFold(n_splits=3, shuffle=True, random_state=42)
    metrics_c = {'Accuracy': [], 'Precision': [], 'Recall': [], 'F1': []}

    for train_c, test_c in skf_c.split(Xc, yc):
        Xc_train, Xc_test = Xc[train_c], Xc[test_c]
        yc_train, yc_test = yc[train_c], yc[test_c]

        clf = GNB()
        clf.fit(Xc_train, yc_train)
        y_pred_c = clf.predict(Xc_test)

        metrics_c['Accuracy'].append(accuracy_score(yc_test, y_pred_c))
        metrics_c['Precision'].append(precision_score(yc_test, y_pred_c))
        metrics_c['Recall'].append(recall_score(yc_test, y_pred_c))
        metrics_c['F1'].append(f1_score(yc_test, y_pred_c))

    # average metrics for this cluster
    cluster_summaries[f'Cluster {c}'] = {
        metric: np.mean(scores)
        for metric, scores in metrics_c.items()
    }

# Compile and display comparison
results = pd.DataFrame({
    'Global': global_summary,
    **cluster_summaries
})

print("=== Classification Performance: Global vs. Cluster-wise ===")
print(results.round(4))